//+------------------------------------------------------------------+
//|                                    Hunter_EA_v2_Combined.mq5     |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "2.10"
#property description "Hunter EA - Combined FVG & Liquidity Trading System"
#property description "Trades Fair Value Gaps aligned with daily market bias"
#property description "All libraries embedded - no external dependencies"

//+------------------------------------------------------------------+
//| FVG LIBRARY SECTION                                              |
//+------------------------------------------------------------------+

//--- Structure to store FVG information
struct FVGInfo
{
   int fvg_id;
   int middle_candle_bar;
   datetime candle_time;
   double top_price;
   double bottom_price;
   double enc_point;
   double gap_size;
   bool is_bullish;
   bool is_active;
   bool enc_touched;
   string directional_bias;
   ulong trade_ticket;
   bool trade_opened;
   datetime enc_time;
   bool is_inverse;
   bool is_reset;
};

//--- FVG Callback type
typedef void (*FVGCallback)(FVGInfo& fvg, string event_type);

//--- FVG Library Class
class CFVGLibrary
{
private:
   FVGInfo           m_fvg_list[];
   int               m_fvg_count;
   int               m_next_fvg_id;
   int               m_bars_total;
   string            m_symbol;
   ENUM_TIMEFRAMES   m_timeframe;
   ENUM_TIMEFRAMES   m_accuracy_tf;
   double            m_min_gap_size;
   bool              m_use_accuracy;
   FVGCallback       m_callback;
   bool              m_draw_objects;
   
public:
   CFVGLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5, 
               ENUM_TIMEFRAMES accuracy_tf = PERIOD_M1, double min_gap_size = 50.0,
               bool use_accuracy = false, bool draw_objects = true);
   ~CFVGLibrary();
   
   bool              Init();
   void              Deinit();
   void              OnTick();
   void              OnNewBar();
   void              SetCallback(FVGCallback callback) { m_callback = callback; }
   void              SetMinGapSize(double gap_size) { m_min_gap_size = gap_size; }
   void              SetAccuracy(bool use_accuracy) { m_use_accuracy = use_accuracy; }
   void              SetDrawObjects(bool draw) { m_draw_objects = draw; }
   void              CheckForNewFVG();
   void              CheckFVGEncroachment();
   void              CheckFVGEncroachmentHighAccuracy();
   void              CheckForInverseFVG();
   bool              CreateFVG(int middle_bar, datetime candle_time, double top, 
                              double bottom, double gap_size, bool is_bullish);
   int               GetFVGCount() const { return m_fvg_count; }
   int               GetActiveFVGsCount() const;
   FVGInfo           GetFVGByIndex(int index);
   FVGInfo           GetFVGById(int fvg_id);
   void              DeactivateFVG(int fvg_id);
   void              AssociateTradeWithFVG(int fvg_id, ulong ticket);
   void              PlotFVGDot(int bar_index, datetime candle_time, bool is_bullish, int fvg_id);
   void              PlotEncroachmentDot(int fvg_id, string bias);
   void              CleanupObjects();
   void              PrintFVGStats() const;
   double            GetFillRate() const;
};

CFVGLibrary::CFVGLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5, 
                        ENUM_TIMEFRAMES accuracy_tf = PERIOD_M1, double min_gap_size = 50.0,
                        bool use_accuracy = false, bool draw_objects = true)
{
   m_symbol = (symbol == "") ? _Symbol : symbol;
   m_timeframe = timeframe;
   m_accuracy_tf = accuracy_tf;
   m_min_gap_size = min_gap_size;
   m_use_accuracy = use_accuracy;
   m_draw_objects = draw_objects;
   m_fvg_count = 0;
   m_next_fvg_id = 1;
   m_bars_total = 0;
   m_callback = NULL;
   ArrayResize(m_fvg_list, 100);
}

CFVGLibrary::~CFVGLibrary() { Deinit(); }

bool CFVGLibrary::Init()
{
   Print("=== FVG Library Initialized ===");
   Print("Symbol: ", m_symbol);
   Print("Timeframe: ", EnumToString(m_timeframe));
   Print("Minimum Gap Size: ", m_min_gap_size, " points");
   Print("Accuracy Mode: ", (m_use_accuracy ? "ON" : "OFF"));
   
   if(m_draw_objects) CleanupObjects();
   m_fvg_count = 0;
   m_next_fvg_id = 1;
   m_bars_total = iBars(m_symbol, m_timeframe);
   return true;
}

void CFVGLibrary::Deinit()
{
   if(m_draw_objects) CleanupObjects();
   Print("=== FVG Library Deinitialized ===");
   Print("Total FVGs detected: ", m_fvg_count);
}

void CFVGLibrary::OnTick()
{
   int current_bars = iBars(m_symbol, m_timeframe);
   if(current_bars > m_bars_total)
   {
      m_bars_total = current_bars;
      OnNewBar();
   }
   CheckForInverseFVG();
   if(m_use_accuracy) CheckFVGEncroachmentHighAccuracy();
   else CheckFVGEncroachment();
}

void CFVGLibrary::OnNewBar() { CheckForNewFVG(); }

void CFVGLibrary::CheckForNewFVG()
{
   int bars_available = iBars(m_symbol, m_timeframe);
   if(bars_available < 4) return;
   
   double high1 = iHigh(m_symbol, m_timeframe, 3);
   double low1 = iLow(m_symbol, m_timeframe, 3);
   double high2 = iHigh(m_symbol, m_timeframe, 2);
   double low2 = iLow(m_symbol, m_timeframe, 2);
   double high3 = iHigh(m_symbol, m_timeframe, 1);
   double low3 = iLow(m_symbol, m_timeframe, 1);
   datetime middle_candle_time = iTime(m_symbol, m_timeframe, 2);
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   double min_gap_price = m_min_gap_size * point;
   
   if(high1 < low3)
   {
      double gap_size = low3 - high1;
      if(gap_size >= min_gap_price)
         CreateFVG(2, middle_candle_time, low3, high1, gap_size, true);
   }
   else if(low1 > high3)
   {
      double gap_size = low1 - high3;
      if(gap_size >= min_gap_price)
         CreateFVG(2, middle_candle_time, low1, high3, gap_size, false);
   }
}

bool CFVGLibrary::CreateFVG(int middle_bar, datetime candle_time, double top, 
                           double bottom, double gap_size, bool is_bullish)
{
   if(m_fvg_count >= ArraySize(m_fvg_list))
      ArrayResize(m_fvg_list, ArraySize(m_fvg_list) + 50);
   
   double enc_point = (top + bottom) / 2.0;
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   
   m_fvg_list[m_fvg_count].fvg_id = m_next_fvg_id;
   m_fvg_list[m_fvg_count].middle_candle_bar = middle_bar;
   m_fvg_list[m_fvg_count].candle_time = candle_time;
   m_fvg_list[m_fvg_count].top_price = top;
   m_fvg_list[m_fvg_count].bottom_price = bottom;
   m_fvg_list[m_fvg_count].enc_point = enc_point;
   m_fvg_list[m_fvg_count].gap_size = gap_size / point;
   m_fvg_list[m_fvg_count].is_bullish = is_bullish;
   m_fvg_list[m_fvg_count].is_active = true;
   m_fvg_list[m_fvg_count].enc_touched = false;
   m_fvg_list[m_fvg_count].directional_bias = "PENDING";
   m_fvg_list[m_fvg_count].trade_ticket = 0;
   m_fvg_list[m_fvg_count].trade_opened = false;
   m_fvg_list[m_fvg_count].enc_time = 0;
   m_fvg_list[m_fvg_count].is_inverse = false;
   m_fvg_list[m_fvg_count].is_reset = false;
   
   if(m_draw_objects) PlotFVGDot(middle_bar, candle_time, is_bullish, m_next_fvg_id);
   if(m_callback != NULL) m_callback(m_fvg_list[m_fvg_count], "FVG_FORMED");
   
   Print(">>> FVG ", m_next_fvg_id, " formed! Type: ", (is_bullish ? "BULLISH" : "BEARISH"));
   
   m_fvg_count++;
   m_next_fvg_id++;
   return true;
}

void CFVGLibrary::CheckFVGEncroachment()
{
   double current_high = iHigh(m_symbol, m_timeframe, 0);
   double current_low = iLow(m_symbol, m_timeframe, 0);
   double current_close = iClose(m_symbol, m_timeframe, 0);
   
   for(int i = 0; i < m_fvg_count; i++)
   {
      if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched) continue;
      
      bool enc_touched_now = false;
      string bias = "PENDING";
      
      if(m_fvg_list[i].is_bullish)
      {
         if(current_low <= m_fvg_list[i].enc_point && current_high >= m_fvg_list[i].enc_point)
         {
            enc_touched_now = true;
            bias = (current_close > m_fvg_list[i].enc_point) ? "BULLISH" : "BEARISH";
         }
      }
      else
      {
         if(current_high >= m_fvg_list[i].enc_point && current_low <= m_fvg_list[i].enc_point)
         {
            enc_touched_now = true;
            bias = (current_close < m_fvg_list[i].enc_point) ? "BEARISH" : "BULLISH";
         }
      }
      
      if(enc_touched_now)
      {
         m_fvg_list[i].enc_touched = true;
         m_fvg_list[i].is_active = false;
         m_fvg_list[i].directional_bias = bias;
         m_fvg_list[i].enc_time = TimeCurrent();
         if(m_draw_objects) PlotEncroachmentDot(m_fvg_list[i].fvg_id, bias);
         if(m_callback != NULL) m_callback(m_fvg_list[i], "FVG_ENCROACHED");
         Print("*** FVG ", m_fvg_list[i].fvg_id, " ENCROACHED! Bias: ", bias);
      }
   }
}

void CFVGLibrary::CheckFVGEncroachmentHighAccuracy()
{
   double current_high = iHigh(m_symbol, m_accuracy_tf, 0);
   double current_low = iLow(m_symbol, m_accuracy_tf, 0);
   double current_close = iClose(m_symbol, m_accuracy_tf, 0);
   
   for(int i = 0; i < m_fvg_count; i++)
   {
      if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched) continue;
      
      bool enc_touched_now = false;
      string bias = "PENDING";
      
      if(m_fvg_list[i].is_bullish)
      {
         if(current_low <= m_fvg_list[i].enc_point && current_high >= m_fvg_list[i].enc_point)
         {
            enc_touched_now = true;
            bias = (current_close > m_fvg_list[i].enc_point) ? "BULLISH" : "BEARISH";
         }
      }
      else
      {
         if(current_high >= m_fvg_list[i].enc_point && current_low <= m_fvg_list[i].enc_point)
         {
            enc_touched_now = true;
            bias = (current_close < m_fvg_list[i].enc_point) ? "BEARISH" : "BULLISH";
         }
      }
      
      if(enc_touched_now)
      {
         m_fvg_list[i].enc_touched = true;
         m_fvg_list[i].is_active = false;
         m_fvg_list[i].directional_bias = bias;
         m_fvg_list[i].enc_time = TimeCurrent();
         if(m_draw_objects) PlotEncroachmentDot(m_fvg_list[i].fvg_id, bias);
         if(m_callback != NULL) m_callback(m_fvg_list[i], "FVG_ENCROACHED_HA");
         Print("*** FVG ", m_fvg_list[i].fvg_id, " ENCROACHED (HA)! Bias: ", bias);
      }
   }
}

void CFVGLibrary::CheckForInverseFVG()
{
   double current_open = iOpen(m_symbol, m_timeframe, 0);
   double current_close = iClose(m_symbol, m_timeframe, 0);
   
   for(int i = 0; i < m_fvg_count; i++)
   {
      if(!m_fvg_list[i].is_active || m_fvg_list[i].enc_touched || m_fvg_list[i].is_reset) continue;
      
      bool is_inverse = false;
      
      if(m_fvg_list[i].is_bullish)
      {
         if(current_open < m_fvg_list[i].bottom_price && current_close > m_fvg_list[i].top_price)
            is_inverse = true;
      }
      else
      {
         if(current_open > m_fvg_list[i].top_price && current_close < m_fvg_list[i].bottom_price)
            is_inverse = true;
      }
      
      if(is_inverse)
      {
         m_fvg_list[i].is_reset = true;
         m_fvg_list[i].is_active = false;
         m_fvg_list[i].directional_bias = "RESET";
         m_fvg_list[i].enc_time = TimeCurrent();
         m_fvg_list[i].is_inverse = true;
         if(m_callback != NULL) m_callback(m_fvg_list[i], "FVG_RESET");
         Print("*** FVG ", m_fvg_list[i].fvg_id, " RESET by INVERSE CANDLE! ***");
      }
   }
}

int CFVGLibrary::GetActiveFVGsCount() const
{
   int active_count = 0;
   for(int i = 0; i < m_fvg_count; i++)
      if(m_fvg_list[i].is_active && !m_fvg_list[i].enc_touched)
         active_count++;
   return active_count;
}

FVGInfo CFVGLibrary::GetFVGByIndex(int index)
{
   FVGInfo empty_fvg = {0};
   if(index >= 0 && index < m_fvg_count)
      return m_fvg_list[index];
   return empty_fvg;
}

FVGInfo CFVGLibrary::GetFVGById(int fvg_id)
{
   FVGInfo empty_fvg = {0};
   for(int i = 0; i < m_fvg_count; i++)
      if(m_fvg_list[i].fvg_id == fvg_id)
         return m_fvg_list[i];
   return empty_fvg;
}

void CFVGLibrary::DeactivateFVG(int fvg_id)
{
   for(int i = 0; i < m_fvg_count; i++)
   {
      if(m_fvg_list[i].fvg_id == fvg_id)
      {
         m_fvg_list[i].is_active = false;
         if(m_callback != NULL) m_callback(m_fvg_list[i], "FVG_DEACTIVATED");
         break;
      }
   }
}

void CFVGLibrary::AssociateTradeWithFVG(int fvg_id, ulong ticket)
{
   for(int i = 0; i < m_fvg_count; i++)
   {
      if(m_fvg_list[i].fvg_id == fvg_id)
      {
         m_fvg_list[i].trade_ticket = ticket;
         m_fvg_list[i].trade_opened = true;
         break;
      }
   }
}

void CFVGLibrary::PlotFVGDot(int bar_index, datetime candle_time, bool is_bullish, int fvg_id)
{
   if(!m_draw_objects) return;
   string obj_name = "FVG_DOT_" + IntegerToString(fvg_id);
   double candle_high = iHigh(m_symbol, m_timeframe, bar_index);
   double candle_low = iLow(m_symbol, m_timeframe, bar_index);
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   double dot_price = is_bullish ? candle_high + (10 * point) : candle_low - (10 * point);
   color dot_color = is_bullish ? clrLimeGreen : clrRed;
   
   ObjectCreate(0, obj_name, OBJ_ARROW, 0, candle_time, dot_price);
   ObjectSetInteger(0, obj_name, OBJPROP_ARROWCODE, 159);
   ObjectSetInteger(0, obj_name, OBJPROP_COLOR, dot_color);
   ObjectSetInteger(0, obj_name, OBJPROP_WIDTH, 3);
}

void CFVGLibrary::PlotEncroachmentDot(int fvg_id, string bias)
{
   if(!m_draw_objects) return;
   string obj_name = "ENC_DOT_" + IntegerToString(fvg_id);
   datetime current_time = iTime(m_symbol, m_timeframe, 0);
   double candle_high = iHigh(m_symbol, m_timeframe, 0);
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   double dot_price = candle_high + (15 * point);
   color dot_color = (bias == "BULLISH") ? clrOrange : clrBlue;
   
   ObjectCreate(0, obj_name, OBJ_ARROW, 0, current_time, dot_price);
   ObjectSetInteger(0, obj_name, OBJPROP_ARROWCODE, 159);
   ObjectSetInteger(0, obj_name, OBJPROP_COLOR, dot_color);
   ObjectSetInteger(0, obj_name, OBJPROP_WIDTH, 3);
}

void CFVGLibrary::CleanupObjects()
{
   ObjectsDeleteAll(0, "FVG_DOT_");
   ObjectsDeleteAll(0, "ENC_DOT_");
}

void CFVGLibrary::PrintFVGStats() const
{
   int active_fvgs = GetActiveFVGsCount();
   int filled_fvgs = m_fvg_count - active_fvgs;
   Print("=== FVG LIBRARY STATISTICS ===");
   Print("Total FVGs detected: ", m_fvg_count);
   Print("Active FVGs: ", active_fvgs);
   Print("Filled FVGs: ", filled_fvgs);
   if(m_fvg_count > 0)
      Print("Fill rate: ", DoubleToString(GetFillRate(), 1), "%");
}

double CFVGLibrary::GetFillRate() const
{
   if(m_fvg_count == 0) return 0.0;
   int filled_count = 0;
   for(int i = 0; i < m_fvg_count; i++)
      if(m_fvg_list[i].enc_touched) filled_count++;
   return (double)filled_count / m_fvg_count * 100.0;
}

//+------------------------------------------------------------------+
//| LIQUIDITY LIBRARY SECTION                                       |
//+------------------------------------------------------------------+

enum ENUM_LIQUIDITY_TYPE
{
   LIQUIDITY_SWING_HIGH, LIQUIDITY_SWING_LOW, LIQUIDITY_EQUAL_HIGHS,
   LIQUIDITY_EQUAL_LOWS, LIQUIDITY_SESSION_HIGH, LIQUIDITY_SESSION_LOW,
   LIQUIDITY_PDH, LIQUIDITY_PDL, LIQUIDITY_PWH, LIQUIDITY_PWL
};

enum ENUM_SESSION_TYPE
{
   SESSION_SYDNEY, SESSION_TOKYO, SESSION_LONDON, SESSION_NEW_YORK,
   SESSION_SILVER_BULLET_1, SESSION_SILVER_BULLET_2, SESSION_SILVER_BULLET_3
};

struct LiquidityInfo
{
   int liquidity_id;
   ENUM_LIQUIDITY_TYPE type;
   datetime time_created;
   double price_level;
   bool is_swept;
   bool is_active;
   datetime sweep_time;
   string session_origin;
   bool is_equal_level;
   int equal_count;
   double tolerance;
};

struct SessionData
{
   ENUM_SESSION_TYPE session_type;
   datetime session_start;
   datetime session_end;
   double session_high;
   double session_low;
   bool is_active;
   string session_name;
};

struct DailyBiasInfo
{
   datetime bias_date;
   double pdh_level;
   double pdl_level;
   double pwh_level;
   double pwl_level;
   string current_dol;
   string daily_bias;
   bool pdh_failure_to_displace;
   bool pdl_failure_to_displace;
};

typedef void (*LiquidityCallback)(LiquidityInfo& liq, string event_type);

class CLiquidityLibrary
{
private:
   LiquidityInfo     m_liquidity_list[];
   SessionData       m_sessions[];
   DailyBiasInfo     m_daily_bias;
   int               m_liquidity_count;
   int               m_next_liq_id;
   string            m_symbol;
   ENUM_TIMEFRAMES   m_timeframe;
   int               m_gmt_offset;
   LiquidityCallback m_callback;
   bool              m_draw_objects;
   double            m_equal_tolerance;
   int               m_swing_lookback;
   
public:
   CLiquidityLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5,
                    int gmt_offset = 0, bool draw_objects = true, double equal_tolerance = 5.0);
   ~CLiquidityLibrary();
   
   bool              Init();
   void              Deinit();
   void              OnTick();
   void              OnNewBar();
   void              SetCallback(LiquidityCallback callback) { m_callback = callback; }
   void              DetectSwingPoints();
   bool              IsSwingHigh(int bar_index);
   bool              IsSwingLow(int bar_index);
   void              UpdateSessions();
   void              CalculateSessionHighLow(ENUM_SESSION_TYPE session_type);
   void              UpdateDailyBias();
   void              CalculatePreviousDayHighLow();
   void              CalculatePreviousWeekHighLow();
   string            DetermineDOL();
   bool              CreateLiquidity(ENUM_LIQUIDITY_TYPE type, double price, datetime time, string session = "");
   void              CheckLiquiditySweeps();
   void              DetectEqualLevels();
   void              UpdateLiquidityStatus();
   int               GetLiquidityCount() const { return m_liquidity_count; }
   string            GetCurrentDOL() const { return m_daily_bias.current_dol; }
   string            GetDailyBias() const { return m_daily_bias.daily_bias; }
   void              PlotLiquidityLevel(LiquidityInfo& liq);
   void              PlotSweptLevel(LiquidityInfo& liq);
   void              CleanupObjects();
   void              PrintDailyBias() const;
   bool              CheckFailureToDisplace(double level, bool is_high);
};

CLiquidityLibrary::CLiquidityLibrary(string symbol = "", ENUM_TIMEFRAMES timeframe = PERIOD_M5,
                                   int gmt_offset = 0, bool draw_objects = true, double equal_tolerance = 5.0)
{
   m_symbol = (symbol == "") ? _Symbol : symbol;
   m_timeframe = timeframe;
   m_gmt_offset = gmt_offset;
   m_draw_objects = draw_objects;
   m_equal_tolerance = equal_tolerance;
   m_liquidity_count = 0;
   m_next_liq_id = 1;
   m_swing_lookback = 3;
   m_callback = NULL;
   ArrayResize(m_liquidity_list, 200);
   ArrayResize(m_sessions, 7);
}

CLiquidityLibrary::~CLiquidityLibrary() { Deinit(); }

bool CLiquidityLibrary::Init()
{
   Print("=== Liquidity Library Initialized ===");
   Print("Symbol: ", m_symbol);
   Print("Timeframe: ", EnumToString(m_timeframe));
   if(m_draw_objects) CleanupObjects();
   UpdateSessions();
   UpdateDailyBias();
   return true;
}

void CLiquidityLibrary::Deinit()
{
   if(m_draw_objects) CleanupObjects();
   Print("=== Liquidity Library Deinitialized ===");
}

void CLiquidityLibrary::OnTick()
{
   UpdateSessions();
   CheckLiquiditySweeps();
   UpdateLiquidityStatus();
}

void CLiquidityLibrary::OnNewBar()
{
   DetectSwingPoints();
   DetectEqualLevels();
   UpdateDailyBias();
}

void CLiquidityLibrary::DetectSwingPoints()
{
   int bars_available = iBars(m_symbol, m_timeframe);
   if(bars_available < m_swing_lookback * 2 + 1) return;
   
   if(IsSwingHigh(1))
   {
      double high_price = iHigh(m_symbol, m_timeframe, 1);
      datetime high_time = iTime(m_symbol, m_timeframe, 1);
      CreateLiquidity(LIQUIDITY_SWING_HIGH, high_price, high_time);
   }
   
   if(IsSwingLow(1))
   {
      double low_price = iLow(m_symbol, m_timeframe, 1);
      datetime low_time = iTime(m_symbol, m_timeframe, 1);
      CreateLiquidity(LIQUIDITY_SWING_LOW, low_price, low_time);
   }
}

bool CLiquidityLibrary::IsSwingHigh(int bar_index)
{
   if(bar_index < m_swing_lookback || bar_index >= iBars(m_symbol, m_timeframe) - m_swing_lookback)
      return false;
   double center_high = iHigh(m_symbol, m_timeframe, bar_index);
   for(int i = 1; i <= m_swing_lookback; i++)
   {
      if(iHigh(m_symbol, m_timeframe, bar_index + i) >= center_high) return false;
      if(iHigh(m_symbol, m_timeframe, bar_index - i) >= center_high) return false;
   }
   return true;
}

bool CLiquidityLibrary::IsSwingLow(int bar_index)
{
   if(bar_index < m_swing_lookback || bar_index >= iBars(m_symbol, m_timeframe) - m_swing_lookback)
      return false;
   double center_low = iLow(m_symbol, m_timeframe, bar_index);
   for(int i = 1; i <= m_swing_lookback; i++)
   {
      if(iLow(m_symbol, m_timeframe, bar_index + i) <= center_low) return false;
      if(iLow(m_symbol, m_timeframe, bar_index - i) <= center_low) return false;
   }
   return true;
}

void CLiquidityLibrary::UpdateSessions()
{
   datetime current_time = TimeCurrent() + (m_gmt_offset * 3600);
   MqlDateTime time_struct;
   TimeToStruct(current_time, time_struct);
   int current_hour = time_struct.hour;
   
   for(int i = 0; i < ArraySize(m_sessions); i++) m_sessions[i].is_active = false;
   
   if((current_hour >= 21) || (current_hour < 6))
   {
      m_sessions[SESSION_SYDNEY].is_active = true;
      m_sessions[SESSION_SYDNEY].session_name = "Sydney";
   }
   if(current_hour >= 0 && current_hour < 9)
   {
      m_sessions[SESSION_TOKYO].is_active = true;
      m_sessions[SESSION_TOKYO].session_name = "Tokyo";
   }
   if(current_hour >= 7 && current_hour < 16)
   {
      m_sessions[SESSION_LONDON].is_active = true;
      m_sessions[SESSION_LONDON].session_name = "London";
   }
}

void CLiquidityLibrary::UpdateDailyBias()
{
   CalculatePreviousDayHighLow();
   CalculatePreviousWeekHighLow();
   m_daily_bias.current_dol = DetermineDOL();
   
   if(StringFind(m_daily_bias.current_dol, "HIGH") != -1)
      m_daily_bias.daily_bias = "BULLISH";
   else if(StringFind(m_daily_bias.current_dol, "LOW") != -1)
      m_daily_bias.daily_bias = "BEARISH";
   else
      m_daily_bias.daily_bias = "NEUTRAL";
}

void CLiquidityLibrary::CalculatePreviousDayHighLow()
{
   m_daily_bias.pdh_level = iHigh(m_symbol, PERIOD_D1, 1);
   m_daily_bias.pdl_level = iLow(m_symbol, PERIOD_D1, 1);
   datetime yesterday = iTime(m_symbol, PERIOD_D1, 1);
   CreateLiquidity(LIQUIDITY_PDH, m_daily_bias.pdh_level, yesterday);
   CreateLiquidity(LIQUIDITY_PDL, m_daily_bias.pdl_level, yesterday);
}

void CLiquidityLibrary::CalculatePreviousWeekHighLow()
{
   m_daily_bias.pwh_level = iHigh(m_symbol, PERIOD_W1, 1);
   m_daily_bias.pwl_level = iLow(m_symbol, PERIOD_W1, 1);
   datetime last_week = iTime(m_symbol, PERIOD_W1, 1);
   CreateLiquidity(LIQUIDITY_PWH, m_daily_bias.pwh_level, last_week);
   CreateLiquidity(LIQUIDITY_PWL, m_daily_bias.pwl_level, last_week);
}

string CLiquidityLibrary::DetermineDOL()
{
   double current_open = iOpen(m_symbol, PERIOD_D1, 0);
   double pdh_distance = MathAbs(current_open - m_daily_bias.pdh_level);
   double pdl_distance = MathAbs(current_open - m_daily_bias.pdl_level);
   return (pdh_distance < pdl_distance) ? "PDH" : "PDL";
}

bool CLiquidityLibrary::CreateLiquidity(ENUM_LIQUIDITY_TYPE type, double price, datetime time, string session = "")
{
   if(m_liquidity_count >= ArraySize(m_liquidity_list))
      ArrayResize(m_liquidity_list, ArraySize(m_liquidity_list) + 100);
   
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   double tolerance = m_equal_tolerance * point;
   
   for(int i = 0; i < m_liquidity_count; i++)
      if(m_liquidity_list[i].type == type && MathAbs(m_liquidity_list[i].price_level - price) < tolerance)
         return false;
   
   m_liquidity_list[m_liquidity_count].liquidity_id = m_next_liq_id;
   m_liquidity_list[m_liquidity_count].type = type;
   m_liquidity_list[m_liquidity_count].time_created = time;
   m_liquidity_list[m_liquidity_count].price_level = price;
   m_liquidity_list[m_liquidity_count].is_swept = false;
   m_liquidity_list[m_liquidity_count].is_active = true;
   m_liquidity_list[m_liquidity_count].session_origin = session;
   
   if(m_draw_objects) PlotLiquidityLevel(m_liquidity_list[m_liquidity_count]);
   if(m_callback != NULL) m_callback(m_liquidity_list[m_liquidity_count], "LIQUIDITY_CREATED");
   
   m_liquidity_count++;
   m_next_liq_id++;
   return true;
}

void CLiquidityLibrary::CheckLiquiditySweeps()
{
   double current_high = iHigh(m_symbol, m_timeframe, 0);
   double current_low = iLow(m_symbol, m_timeframe, 0);
   
   for(int i = 0; i < m_liquidity_count; i++)
   {
      if(!m_liquidity_list[i].is_active || m_liquidity_list[i].is_swept) continue;
      bool swept = false;
      
      if((m_liquidity_list[i].type == LIQUIDITY_SWING_HIGH || m_liquidity_list[i].type == LIQUIDITY_PDH) && 
         current_high > m_liquidity_list[i].price_level)
         swept = true;
      else if((m_liquidity_list[i].type == LIQUIDITY_SWING_LOW || m_liquidity_list[i].type == LIQUIDITY_PDL) && 
              current_low < m_liquidity_list[i].price_level)
         swept = true;
      
      if(swept)
      {
         m_liquidity_list[i].is_swept = true;
         m_liquidity_list[i].sweep_time = TimeCurrent();
         if(m_draw_objects) PlotSweptLevel(m_liquidity_list[i]);
         if(m_callback != NULL) m_callback(m_liquidity_list[i], "LIQUIDITY_SWEPT");
      }
   }
}

void CLiquidityLibrary::DetectEqualLevels()
{
   double point = SymbolInfoDouble(m_symbol, SYMBOL_POINT);
   double tolerance = m_equal_tolerance * point;
   
   for(int i = 0; i < m_liquidity_count - 1; i++)
   {
      if(!m_liquidity_list[i].is_active) continue;
      for(int j = i + 1; j < m_liquidity_count; j++)
      {
         if(!m_liquidity_list[j].is_active) continue;
         if(MathAbs(m_liquidity_list[i].price_level - m_liquidity_list[j].price_level) < tolerance)
         {
            m_liquidity_list[i].is_equal_level = true;
            m_liquidity_list[j].is_equal_level = true;
         }
      }
   }
}

void CLiquidityLibrary::UpdateLiquidityStatus()
{
   m_daily_bias.pdh_failure_to_displace = CheckFailureToDisplace(m_daily_bias.pdh_level, true);
   m_daily_bias.pdl_failure_to_displace = CheckFailureToDisplace(m_daily_bias.pdl_level, false);
   
   if(m_daily_bias.pdh_failure_to_displace)
      m_daily_bias.current_dol = "PDL";
   else if(m_daily_bias.pdl_failure_to_displace)
      m_daily_bias.current_dol = "PDH";
}

bool CLiquidityLibrary::CheckFailureToDisplace(double level, bool is_high)
{
   double current_high = iHigh(m_symbol, m_timeframe, 0);
   double current_low = iLow(m_symbol, m_timeframe, 0);
   double current_close = iClose(m_symbol, m_timeframe, 0);
   
   if(is_high) return (current_high > level && current_close < level);
   else return (current_low < level && current_close > level);
}

void CLiquidityLibrary::PlotLiquidityLevel(LiquidityInfo& liq)
{
   if(!m_draw_objects) return;
   string obj_name = "LIQ_" + IntegerToString(liq.liquidity_id);
   color line_color = (liq.type == LIQUIDITY_SWING_HIGH || liq.type == LIQUIDITY_PDH) ? clrRed : clrLimeGreen;
   ObjectCreate(0, obj_name, OBJ_HLINE, 0, 0, liq.price_level);
   ObjectSetInteger(0, obj_name, OBJPROP_COLOR, line_color);
   ObjectSetInteger(0, obj_name, OBJPROP_STYLE, STYLE_DASH);
}

void CLiquidityLibrary::PlotSweptLevel(LiquidityInfo& liq)
{
   if(!m_draw_objects) return;
   string obj_name = "LIQ_" + IntegerToString(liq.liquidity_id);
   ObjectSetInteger(0, obj_name, OBJPROP_COLOR, clrGray);
   ObjectSetInteger(0, obj_name, OBJPROP_STYLE, STYLE_DOT);
}

void CLiquidityLibrary::CleanupObjects() { ObjectsDeleteAll(0, "LIQ_"); }

void CLiquidityLibrary::PrintDailyBias() const
{
   Print("=== DAILY BIAS ANALYSIS ===");
   Print("Current DOL: ", m_daily_bias.current_dol);
   Print("Daily Bias: ", m_daily_bias.daily_bias);
   Print("PDH: ", m_daily_bias.pdh_level, " | PDL: ", m_daily_bias.pdl_level);
}

void CLiquidityLibrary::CalculateSessionHighLow(ENUM_SESSION_TYPE session_type)
{
   if(m_sessions[session_type].is_active)
   {
      double current_high = iHigh(m_symbol, m_timeframe, 0);
      double current_low = iLow(m_symbol, m_timeframe, 0);
      if(m_sessions[session_type].session_high < current_high || m_sessions[session_type].session_high == 0)
         m_sessions[session_type].session_high = current_high;
      if(m_sessions[session_type].session_low > current_low || m_sessions[session_type].session_low == 0)
         m_sessions[session_type].session_low = current_low;
   }
}

//+------------------------------------------------------------------+
//| MAIN EA CODE                                                     |
//+------------------------------------------------------------------+

input group "=== Trading Settings ==="
input double InpLotSize = 0.01;
input bool InpUseAutoLotSize = false;
input double InpRiskPercent = 1.0;
input double InpProfitTarget = 2.0;
input bool InpUseStopLoss = true;
input double InpStopLoss = 2.0;
input double InpMaxSpreadPoints = 50.0;

input group "=== Margin & Safety Settings ==="
input double InpMinFreeMarginPercent = 30.0;
input double InpMaxLotSize = 10.0;
input double InpMinLotSize = 0.01;

input group "=== FVG Detection Settings ==="
input ENUM_TIMEFRAMES InpFVGTimeframe = PERIOD_M5;
input double InpMinGapSize = 50.0;
input bool InpDrawFVGObjects = true;

input group "=== Trading Mode ==="
input bool InpUseEncroachmentMode = true;

input group "=== Library Settings ==="
input int InpGMTOffset = 0;

CFVGLibrary* fvg_lib;
CLiquidityLibrary* liq_lib;
string daily_bias = "NEUTRAL";
ulong current_ticket = 0;
int last_traded_fvg_id = -1;

const int EA_MAGIC = 8889;

double point_value, min_lot, max_lot, lot_step;
int digits;

void OnFVGEvent(FVGInfo& fvg, string event_type);

int OnInit()
{
   Print("=== Hunter EA v2 Combined Initialized ===");
   
   min_lot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   max_lot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
   lot_step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   digits = (int)SymbolInfoInteger(_Symbol, SYMBOL_DIGITS);
   point_value = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   
   fvg_lib = new CFVGLibrary(_Symbol, InpFVGTimeframe, PERIOD_M1, InpMinGapSize, false, InpDrawFVGObjects);
   if(!fvg_lib.Init()) return(INIT_FAILED);
   fvg_lib.SetCallback(OnFVGEvent);
   
   liq_lib = new CLiquidityLibrary(_Symbol, PERIOD_M5, InpGMTOffset, false);
   if(!liq_lib.Init()) return(INIT_FAILED);
   
   daily_bias = liq_lib.GetDailyBias();
   if(daily_bias == "NEUTRAL")
   {
      string dol = liq_lib.GetCurrentDOL();
      if(dol == "PDH" || StringFind(dol, "HIGH") != -1) daily_bias = "BULLISH";
      else if(dol == "PDL" || StringFind(dol, "LOW") != -1) daily_bias = "BEARISH";
   }
   
   Print("=== CONFIGURATION ===");
   Print("Encroachment Mode: ", (InpUseEncroachmentMode ? "ON" : "OFF"));
   Print("Auto Lot Sizing: ", (InpUseAutoLotSize ? "ON" : "OFF"));
   Print("Daily Bias: ", daily_bias);
   
   return(INIT_SUCCEEDED);
}

void OnDeinit(const int reason)
{
   if(fvg_lib != NULL) { fvg_lib.PrintFVGStats(); delete fvg_lib; }
   if(liq_lib != NULL) { liq_lib.PrintDailyBias(); delete liq_lib; }
}

void OnTick()
{
   if(fvg_lib != NULL) fvg_lib.OnTick();
   if(liq_lib != NULL) liq_lib.OnTick();
   
   daily_bias = liq_lib.GetDailyBias();
   if(daily_bias == "NEUTRAL")
   {
      string dol = liq_lib.GetCurrentDOL();
      if(StringFind(dol, "HIGH") != -1) daily_bias = "BULLISH";
      else if(StringFind(dol, "LOW") != -1) daily_bias = "BEARISH";
   }
   
   if(current_ticket == 0 || !PositionSelectByTicket(current_ticket))
      if(!InpUseEncroachmentMode) CheckFVGEntry();
   
   ManagePosition();
}

void OnFVGEvent(FVGInfo& fvg, string event_type)
{
   if(event_type == "FVG_ENCROACHED" || event_type == "FVG_ENCROACHED_HA")
   {
      if(InpUseEncroachmentMode) HandleEncroachmentTrade(fvg);
   }
}

void CheckFVGEntry()
{
   if(daily_bias == "NEUTRAL") return;
   double current_high = iHigh(_Symbol, InpFVGTimeframe, 0);
   double current_low = iLow(_Symbol, InpFVGTimeframe, 0);
   
   for(int i = 0; i < fvg_lib.GetFVGCount(); i++)
   {
      FVGInfo fvg = fvg_lib.GetFVGByIndex(i);
      if(!fvg.is_active || fvg.enc_touched || fvg.fvg_id == last_traded_fvg_id) continue;
      
      bool candle_in_fvg = false;
      if(fvg.is_bullish && current_low <= fvg.top_price && current_low >= fvg.bottom_price) candle_in_fvg = true;
      else if(!fvg.is_bullish && current_high >= fvg.bottom_price && current_high <= fvg.top_price) candle_in_fvg = true;
      
      if(candle_in_fvg)
      {
         bool bias_matches = (daily_bias == "BULLISH" && fvg.is_bullish) || (daily_bias == "BEARISH" && !fvg.is_bullish);
         if(bias_matches) { OpenTradeForFVG(fvg); break; }
      }
   }
}

void HandleEncroachmentTrade(FVGInfo& fvg)
{
   if(current_ticket != 0 && PositionSelectByTicket(current_ticket)) return;
   if(fvg.fvg_id == last_traded_fvg_id) return;
   if(fvg.directional_bias != daily_bias) return;
   OpenTradeForFVG(fvg);
}

double CalculateLotSize()
{
   double lot_size = InpLotSize;
   if(InpUseAutoLotSize)
   {
      double balance = AccountInfoDouble(ACCOUNT_BALANCE);
      double risk_amount = balance * (InpRiskPercent / 100.0);
      if(InpUseStopLoss && InpStopLoss > 0) lot_size = risk_amount / InpStopLoss;
      else lot_size = risk_amount / 100.0;
   }
   return NormalizeLotSize(lot_size);
}

double NormalizeLotSize(double lot)
{
   double min_allowed = MathMax(min_lot, InpMinLotSize);
   double max_allowed = MathMin(max_lot, InpMaxLotSize);
   if(lot < min_allowed) lot = min_allowed;
   if(lot > max_allowed) lot = max_allowed;
   lot = MathFloor(lot / lot_step) * lot_step;
   if(lot < min_lot) lot = min_lot;
   if(lot > max_lot) lot = max_lot;
   return NormalizeDouble(lot, 2);
}

bool CheckMarginRequirement(double lot_size, ENUM_ORDER_TYPE order_type)
{
   double free_margin = AccountInfoDouble(ACCOUNT_MARGIN_FREE);
   double equity = AccountInfoDouble(ACCOUNT_EQUITY);
   double required_margin = 0;
   
   if(!OrderCalcMargin(order_type, _Symbol, lot_size, SymbolInfoDouble(_Symbol, SYMBOL_ASK), required_margin))
      return false;
   
   if(required_margin > free_margin) return false;
   double margin_after = free_margin - required_margin;
   double margin_percent = (margin_after / equity) * 100.0;
   return (margin_percent >= InpMinFreeMarginPercent);
}

bool CheckSpread()
{
   long spread_points = SymbolInfoInteger(_Symbol, SYMBOL_SPREAD);
   return (spread_points <= InpMaxSpreadPoints);
}

void OpenTradeForFVG(FVGInfo& fvg)
{
   if(!CheckSpread()) return;
   double lot_size = CalculateLotSize();
   ulong ticket = 0;
   
   if(daily_bias == "BULLISH")
   {
      if(!CheckMarginRequirement(lot_size, ORDER_TYPE_BUY)) return;
      ticket = OpenBuy(lot_size);
   }
   else if(daily_bias == "BEARISH")
   {
      if(!CheckMarginRequirement(lot_size, ORDER_TYPE_SELL)) return;
      ticket = OpenSell(lot_size);
   }
   
   if(ticket > 0)
   {
      current_ticket = ticket;
      last_traded_fvg_id = fvg.fvg_id;
      fvg_lib.AssociateTradeWithFVG(fvg.fvg_id, ticket);
      Print("=== TRADE OPENED === Ticket: ", ticket, " | Direction: ", daily_bias, " | Lot: ", lot_size);
   }
}

void ManagePosition()
{
   if(current_ticket == 0) return;
   if(!PositionSelectByTicket(current_ticket)) { current_ticket = 0; return; }
   
   double profit = PositionGetDouble(POSITION_PROFIT);
   
   if(profit >= InpProfitTarget)
   {
      Print("=== PROFIT TARGET HIT === Profit: $", profit);
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) CloseBuy(current_ticket);
      else CloseSell(current_ticket);
      current_ticket = 0;
   }
   else if(InpUseStopLoss && profit <= -InpStopLoss)
   {
      Print("=== STOP LOSS HIT === Loss: $", profit);
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) CloseBuy(current_ticket);
      else CloseSell(current_ticket);
      current_ticket = 0;
   }
}

ulong OpenBuy(double lotSize)
{
   MqlTradeRequest request; MqlTradeResult result;
   ZeroMemory(request); ZeroMemory(result);
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = lotSize;
   request.type = ORDER_TYPE_BUY;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;
   request.comment = "Hunter_FVG";
   
   if(!OrderSend(request, result) || (result.retcode != TRADE_RETCODE_DONE && result.retcode != TRADE_RETCODE_PLACED))
   {
      if(result.retcode == TRADE_RETCODE_INVALID_FILL)
      {
         request.type_filling = ORDER_FILLING_IOC;
         if(OrderSend(request, result) && result.deal > 0) return result.deal;
      }
      return 0;
   }
   return result.deal;
}

ulong OpenSell(double lotSize)
{
   MqlTradeRequest request; MqlTradeResult result;
   ZeroMemory(request); ZeroMemory(result);
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = lotSize;
   request.type = ORDER_TYPE_SELL;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;
   request.comment = "Hunter_FVG";
   
   if(!OrderSend(request, result) || (result.retcode != TRADE_RETCODE_DONE && result.retcode != TRADE_RETCODE_PLACED))
   {
      if(result.retcode == TRADE_RETCODE_INVALID_FILL)
      {
         request.type_filling = ORDER_FILLING_IOC;
         if(OrderSend(request, result) && result.deal > 0) return result.deal;
      }
      return 0;
   }
   return result.deal;
}

void CloseBuy(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;
   MqlTradeRequest request; MqlTradeResult result;
   ZeroMemory(request); ZeroMemory(result);
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = PositionGetDouble(POSITION_VOLUME);
   request.type = ORDER_TYPE_SELL;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      Print("Buy position closed successfully");
   }
   else
   {
      Print("Failed to close buy position. Error: ", GetLastError());
      Print("Return code: ", result.retcode, " - ", result.comment);
   }
}

void CloseSell(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;
   MqlTradeRequest request; MqlTradeResult result;
   ZeroMemory(request); ZeroMemory(result);
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = PositionGetDouble(POSITION_VOLUME);
   request.type = ORDER_TYPE_BUY;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      Print("Sell position closed successfully");
   }
   else
   {
      Print("Failed to close sell position. Error: ", GetLastError());
      Print("Return code: ", result.retcode, " - ", result.comment);
   }
}
//+------------------------------------------------------------------+